/*
 * Decompiled with CFR 0.152.
 */
package io.dataintell.visionapi.scanner;

import io.dataintell.visionapi.domain.File;
import io.dataintell.visionapi.domain.PaginatedResult;
import io.dataintell.visionapi.entity.VolumeSetting;
import io.dataintell.visionapi.file.BulkProcessorFactory;
import io.dataintell.visionapi.file.FileService;
import io.dataintell.visionapi.file.IngestFileProcessor;
import io.dataintell.visionapi.legacyscanner.ScanQueue;
import io.dataintell.visionapi.service.DirectoryService;
import io.dataintell.visionapi.service.IndexService;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class DeletedFilesManager {
    private final FileService fileService;
    private final DirectoryService directoryService;
    private final IndexService indexService;
    private final BulkProcessorFactory bulkProcessorFactory;
    private ScanQueue deletedQueue;
    private final Logger logger = LoggerFactory.getLogger(DeletedFilesManager.class);

    public DeletedFilesManager(FileService fileService, DirectoryService directoryService, IndexService indexService, BulkProcessorFactory bulkProcessorFactory) {
        this.fileService = fileService;
        this.directoryService = directoryService;
        this.indexService = indexService;
        this.bulkProcessorFactory = bulkProcessorFactory;
    }

    public void tagDeletedFilesForVolumesAfterFullScan(List<Long> volumeIds, Date scanDate, LocalDate indexingDate) {
        this.fileService.tagDeletedFilesByVolumesAndOlderThanIndexingDate(volumeIds, scanDate, indexingDate);
    }

    public void tagDeletedFilesForVolumesAfterDeltaScan(List<VolumeSetting> volumeSettings, Date scanDate, LocalDate indexingDate) {
        volumeSettings.forEach(volumeSetting -> this.tagAllDeletedFiles(volumeSetting.getId(), volumeSetting.getPath(), scanDate, indexingDate));
    }

    private void tagAllDeletedFiles(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate) {
        this.logger.info(String.format("About to tag all deleted files for volumeId [%s], mount point [%s], last scan [%s] and indexing date [%s]", volumeId, mountPoint, scanDate, indexingDate));
        try {
            this.tagDeletedFilesOfDeletedFolders(scanDate, volumeId, indexingDate);
            this.tagDeletedFiles(volumeId, mountPoint, scanDate, indexingDate);
        }
        catch (Exception e) {
            this.logger.error(String.format("There was an error while scanning the deleted files for volumeId [%s], mount point [%s], last scan [%s] and indexing date [%s]", volumeId, mountPoint, scanDate, indexingDate), (Throwable)e);
        }
    }

    private void tagDeletedFiles(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate) {
        try {
            this.indexService.refreshFileIndex(indexingDate);
            IngestFileProcessor ingestFileProcessor = new IngestFileProcessor(this.bulkProcessorFactory.makeBulkProcessor(), indexingDate, false);
            this.deletedQueue = new ScanQueue(ingestFileProcessor);
            File directory = this.directoryService.getDirectory("", "/", volumeId, indexingDate);
            if (directory == null) {
                this.logger.warn(String.format("Unable to tag deleted files for volumeId [%s], mount point [%s], scan date [%s] and indexing date [%s]", volumeId, mountPoint, scanDate, indexingDate));
                return;
            }
            Long numberOfDeletedFiles = this.fileService.getNumberOfDeletedFiles(directory.getPath(), directory.getNumberOfFiles(), volumeId, indexingDate);
            this.logger.info(String.format("There is [%s] deleted files for volumeId [%s]", numberOfDeletedFiles, volumeId));
            if (numberOfDeletedFiles == 0L) {
                return;
            }
            this.tagAllChildDirectories("/", volumeId, mountPoint, scanDate, numberOfDeletedFiles, indexingDate);
            ingestFileProcessor.closeIngestProcessor();
            this.logger.info(String.format("All deletes files of volume [%s] were tagged", volumeId));
        }
        catch (Exception e) {
            this.logger.error(String.format("There was an error while flagging deleted files for volumeId [%s]", volumeId), (Throwable)e);
        }
    }

    private void tagAllChildDirectories(String path, Long volumeId, String mountPoint, Date scanDate, Long numberOfDeletedFiles, LocalDate indexingDate) throws IOException {
        Integer size = 100;
        Long deletedFilesInChild = 0L;
        PaginatedResult directories = this.directoryService.getDirectoriesWithScroll(path, volumeId, size, indexingDate);
        while (directories.getElements().size() > 0) {
            deletedFilesInChild = deletedFilesInChild + this.processDirectory(volumeId, mountPoint, scanDate, indexingDate, directories);
            directories = this.directoryService.getNextListWithScroll(directories.getScrollId());
        }
        long deltaOfDeletedFiles = numberOfDeletedFiles - deletedFilesInChild;
        if (deltaOfDeletedFiles > 0L) {
            this.updateDeletedFiles(path, volumeId, mountPoint, scanDate, Long.valueOf(numberOfDeletedFiles - deletedFilesInChild), indexingDate);
        } else if (deltaOfDeletedFiles < 0L) {
            this.logger.warn(String.format("There is an negative number of deleted files [%s] for the path [%s] and for volumeId [%s]", deltaOfDeletedFiles, path, volumeId));
        }
        this.fileService.clearScroll(directories.getScrollId());
    }

    private Long processDirectory(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate, PaginatedResult<File> directories) throws IOException {
        long deletedFilesInChild = 0L;
        for (File directory : directories.getElements()) {
            Long deletedFilesInDirectory = this.fileService.getNumberOfDeletedFiles(directory.getPath(), directory.getNumberOfFiles(), volumeId, indexingDate);
            if (deletedFilesInDirectory == 0L) continue;
            deletedFilesInChild += deletedFilesInDirectory.longValue();
            this.tagAllChildDirectories(directory.getPath(), volumeId, mountPoint, scanDate, deletedFilesInDirectory, indexingDate);
        }
        return deletedFilesInChild;
    }

    private void updateDeletedFiles(String path, Long volumeId, String mountPoint, Date scanDate, Long numberOfDeletedFiles, LocalDate indexingDate) throws IOException {
        Long count = 0L;
        Integer size = 1000;
        PaginatedResult files = this.fileService.getPotentialDeletedFiles(path, volumeId, scanDate, size, indexingDate);
        while (true) {
            for (File file : files.getElements()) {
                Path fullPath = Paths.get(mountPoint, file.getPath(), file.getName());
                if (!Files.notExists(fullPath, new LinkOption[0])) continue;
                Long l = count;
                Long l2 = count = Long.valueOf(count + 1L);
                file.setDeleted(Boolean.valueOf(true));
                this.deletedQueue.addFileToQueue(file);
                if (!count.equals(numberOfDeletedFiles)) continue;
                this.fileService.clearScroll(files.getScrollId());
                return;
            }
            if (files.getSize() < size) {
                Long deletedFilesNotFound = numberOfDeletedFiles - count;
                this.logger.warn(String.format("%s deleted files were not found for path [%s], volumeId [%s], mount point [%s], scan date [%s] and numberOfDeletedFiles [%s]", deletedFilesNotFound, path, volumeId, mountPoint, scanDate, numberOfDeletedFiles));
                this.fileService.clearScroll(files.getScrollId());
                return;
            }
            files = this.fileService.getNextListWithScroll(files.getScrollId());
        }
    }

    private void tagDeletedFilesOfDeletedFolders(Date scanDate, Long volumeId, LocalDate indexingDate) throws IOException {
        this.logger.info(String.format("About to flag files that are in deleted folders with scan date [%s]", scanDate));
        PaginatedResult deletedFolders = this.directoryService.getDeletedFoldersByScanDateWithScroll(scanDate, volumeId, indexingDate);
        this.tagDeletedFolders(volumeId, indexingDate, deletedFolders);
    }

    private void tagDeletedFolders(Long volumeId, LocalDate indexingDate, PaginatedResult<File> deletedFolders) throws IOException {
        if (deletedFolders.getElements().size() > 0) {
            List paths = deletedFolders.getElements().stream().map(File::getPath).collect(Collectors.toList());
            PaginatedResult nextDeletedFolders = this.directoryService.getNextListWithScroll(deletedFolders.getScrollId());
            this.directoryService.setDeletedFoldersAndFilesByPath(paths, volumeId, indexingDate);
            this.tagDeletedFolders(volumeId, indexingDate, nextDeletedFolders);
        } else {
            this.fileService.clearScroll(deletedFolders.getScrollId());
        }
    }
}

