/*
 * Decompiled with CFR 0.152.
 */
package io.dataintell.scanner.service;

import io.dataintell.scanner.domain.File;
import io.dataintell.scanner.domain.PaginatedResult;
import io.dataintell.scanner.file.BulkProcessorFactory;
import io.dataintell.scanner.file.IngestFileProcessor;
import io.dataintell.scanner.mapper.FileMapper;
import io.dataintell.scanner.repository.ESRepository;
import io.dataintell.scanner.utils.HashUtils;
import io.dataintell.scanner.visitor.ScanQueue;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDate;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.search.sort.SortOrder;
import org.elasticsearch.tasks.TaskInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

@Service
public class DeletedFileService {
    private static final Logger log = LoggerFactory.getLogger(DeletedFileService.class);
    private ScanQueue deletedQueue;
    private final BulkProcessorFactory bulkProcessorFactory;
    private final ESRepository esRepository;
    private final FileMapper fileMapper;

    public DeletedFileService(BulkProcessorFactory bulkProcessorFactory, ESRepository esRepository, FileMapper fileMapper) {
        this.bulkProcessorFactory = bulkProcessorFactory;
        this.esRepository = esRepository;
        this.fileMapper = fileMapper;
    }

    public void tagAllDeletedFiles(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate) {
        log.debug("About to tag all deleted files for volumeId [{}], mount point [{}], last scan [{}] and indexing date [{}]", new Object[]{volumeId, mountPoint, scanDate, indexingDate});
        try {
            this.tagDeletedFilesOfDeletedFolders(scanDate, volumeId, indexingDate);
            this.tagDeletedFiles(volumeId, mountPoint, scanDate, indexingDate);
        }
        catch (Exception e) {
            log.error("There was an error while scanning the deleted files for volumeId [{}], mount point [{}], last scan [{}] and indexing date [{}]", new Object[]{volumeId, mountPoint, scanDate, indexingDate, e});
        }
    }

    private void tagDeletedFiles(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate) {
        try {
            IngestFileProcessor ingestFileProcessor = new IngestFileProcessor(this.bulkProcessorFactory.makeBulkProcessor(), indexingDate, false);
            this.esRepository.refreshFileIndex(indexingDate);
            this.deletedQueue = new ScanQueue(ingestFileProcessor);
            File directory = this.getDirectory(volumeId, indexingDate);
            if (directory == null) {
                log.warn("Unable to tag deleted files for volumeId [{}], mount point [{}], scan date [{}] and indexing date [{}]", new Object[]{volumeId, mountPoint, scanDate, indexingDate});
                return;
            }
            Long numberOfDeletedFiles = this.getNumberOfDeletedFiles(directory.getPath(), directory.getNumberOfFiles(), volumeId, indexingDate);
            log.debug("There is [{}] deleted files for volumeId [{}]", (Object)numberOfDeletedFiles, (Object)volumeId);
            if (numberOfDeletedFiles == 0L) {
                return;
            }
            this.tagAllChildDirectories("/", volumeId, mountPoint, scanDate, numberOfDeletedFiles, indexingDate);
            ingestFileProcessor.closeIngestProcessor();
            log.debug("All deletes files of volume [{}] were tagged", (Object)volumeId);
        }
        catch (Exception e) {
            log.error("There was an error while flagging deleted files for volumeId [{}]", (Object)volumeId, (Object)e);
        }
    }

    private File getDirectory(Long volumeId, LocalDate indexingDate) throws IOException {
        String documentId = HashUtils.getDocumentId((String)"", (String)"/", (Long)volumeId, (Boolean)true);
        GetResponse response = this.esRepository.getDirectoryByDocId(documentId, indexingDate);
        return this.fileMapper.getGsonFromJson(response);
    }

    private void tagAllChildDirectories(String path, Long volumeId, String mountPoint, Date scanDate, Long numberOfDeletedFiles, LocalDate indexingDate) throws IOException {
        Integer size = 100;
        Long deletedFilesInChild = 0L;
        PaginatedResult directories = this.getDirectoriesWithScroll(path, volumeId, size, indexingDate);
        while (directories.getElements().size() > 0) {
            deletedFilesInChild = deletedFilesInChild + this.processDirectory(volumeId, mountPoint, scanDate, indexingDate, directories);
            directories = this.getNextListWithScroll(directories.getScrollId());
        }
        long deltaOfDeletedFiles = numberOfDeletedFiles - deletedFilesInChild;
        if (deltaOfDeletedFiles > 0L) {
            this.updateDeletedFiles(path, volumeId, mountPoint, scanDate, Long.valueOf(numberOfDeletedFiles - deletedFilesInChild), indexingDate);
        } else if (deltaOfDeletedFiles < 0L) {
            log.warn("There is an negative number of deleted files [{}] for the path [{}] and for volumeId [{}]", new Object[]{deltaOfDeletedFiles, path, volumeId});
        }
        this.esRepository.clearScroll(directories.getScrollId());
    }

    private Long processDirectory(Long volumeId, String mountPoint, Date scanDate, LocalDate indexingDate, PaginatedResult<File> directories) throws IOException {
        long deletedFilesInChild = 0L;
        for (File directory : directories.getElements()) {
            Long deletedFilesInDirectory = this.getNumberOfDeletedFiles(directory.getPath(), directory.getNumberOfFiles(), volumeId, indexingDate);
            if (deletedFilesInDirectory == 0L) continue;
            deletedFilesInChild += deletedFilesInDirectory.longValue();
            this.tagAllChildDirectories(directory.getPath(), volumeId, mountPoint, scanDate, deletedFilesInDirectory, indexingDate);
        }
        return deletedFilesInChild;
    }

    private void updateDeletedFiles(String path, Long volumeId, String mountPoint, Date scanDate, Long numberOfDeletedFiles, LocalDate indexingDate) throws IOException {
        Long count = 0L;
        Integer size = 1000;
        PaginatedResult files = this.getPotentialDeletedFiles(path, volumeId, scanDate, size, indexingDate);
        while (true) {
            for (File file : files.getElements()) {
                Path fullPath = Paths.get(mountPoint, file.getPath(), file.getName());
                if (!Files.notExists(fullPath, new LinkOption[0])) continue;
                Long l = count;
                Long l2 = count = Long.valueOf(count + 1L);
                file.setIsDeleted(Boolean.valueOf(true));
                this.deletedQueue.addFileToQueue(file);
                if (!count.equals(numberOfDeletedFiles)) continue;
                this.esRepository.clearScroll(files.getScrollId());
                return;
            }
            if (files.getSize() < size) {
                Long deletedFilesNotFound = numberOfDeletedFiles - count;
                log.warn("{} deleted files were not found for path [{}], volumeId [{}], mount point [{}], scan date [{}] and numberOfDeletedFiles [{}]", new Object[]{deletedFilesNotFound, path, volumeId, mountPoint, scanDate, numberOfDeletedFiles});
                this.esRepository.clearScroll(files.getScrollId());
                return;
            }
            files = this.getNextListWithScroll(files.getScrollId());
        }
    }

    private void tagDeletedFilesOfDeletedFolders(Date scanDate, Long volumeId, LocalDate indexingDate) throws IOException {
        log.debug("About to flag files that are in deleted folders with scan date {}", (Object)scanDate);
        PaginatedResult deletedFolders = this.getDeletedFoldersByScanDateWithScroll(scanDate, volumeId, indexingDate);
        this.tagDeletedFolders(volumeId, indexingDate, deletedFolders);
    }

    private void tagDeletedFolders(Long volumeId, LocalDate indexingDate, PaginatedResult<File> deletedFolders) throws IOException {
        if (deletedFolders.getElements().size() > 0) {
            List paths = deletedFolders.getElements().stream().map(File::getPath).collect(Collectors.toList());
            PaginatedResult nextDeletedFolders = this.getNextListWithScroll(deletedFolders.getScrollId());
            this.setDeletedFoldersAndFilesByPath(paths, volumeId, indexingDate);
            this.tagDeletedFolders(volumeId, indexingDate, nextDeletedFolders);
        } else {
            this.esRepository.clearScroll(deletedFolders.getScrollId());
        }
    }

    private Long getNumberOfDeletedFiles(String path, Long numberOfFiles, Long volumeId, LocalDate indexingDate) throws IOException {
        SearchResponse searchResponse = this.esRepository.getFilesOfDirectories(Collections.singletonList(path), Collections.singletonList(volumeId), Boolean.valueOf(false), Boolean.valueOf(false), null, Integer.valueOf(0), Integer.valueOf(0), "size", SortOrder.ASC, Boolean.valueOf(true), indexingDate);
        return searchResponse.getHits().getTotalHits().value - numberOfFiles;
    }

    private PaginatedResult<File> getDirectoriesWithScroll(String path, Long volumeId, Integer size, LocalDate indexingDate) throws IOException {
        SearchResponse response = this.esRepository.getDirectoriesWithScroll(path, volumeId, size, indexingDate);
        return this.fileMapper.getPaginatedResultWithScrollId(response);
    }

    private PaginatedResult<File> getNextListWithScroll(String scrollId) throws IOException {
        SearchResponse response = this.esRepository.getNextListWithScroll(scrollId);
        return this.fileMapper.getPaginatedResultWithScrollId(response);
    }

    private PaginatedResult<File> getPotentialDeletedFiles(String path, Long volumeId, Date scanDate, Integer size, LocalDate indexingDate) throws IOException {
        SearchResponse searchResponse = this.esRepository.getFilesOfDirectoryIndexedBefore(path, volumeId, scanDate, size, indexingDate);
        return this.fileMapper.getPaginatedResultWithScrollId(searchResponse);
    }

    private PaginatedResult<File> getDeletedFoldersByScanDateWithScroll(Date scanDate, Long volumeId, LocalDate indexingDate) throws IOException {
        SearchResponse response = this.esRepository.getDeletedFoldersByScanDateWithScroll(Integer.valueOf(500), scanDate, volumeId, indexingDate);
        return this.fileMapper.getPaginatedResultWithScrollId(response);
    }

    private void setDeletedFoldersAndFilesByPath(List<String> paths, Long volumeId, LocalDate indexingDate) throws IOException {
        try {
            this.esRepository.setDeletedFoldersAndFilesByPath(paths, volumeId, indexingDate);
        }
        catch (SocketTimeoutException e) {
            this.handleLongProcesses();
        }
    }

    private void handleLongProcesses() throws IOException {
        List taskInfos = this.esRepository.getTasksByName("indices:data/write/update/byquery");
        if (taskInfos.get(0) != null) {
            log.debug("Task of type [{}] is still running with status [{}]", (Object)"indices:data/write/update/byquery", (Object)((TaskInfo)taskInfos.get(0)).getStatus());
            try {
                Thread.sleep(5000L);
            }
            catch (InterruptedException e) {
                log.error("Fail to handle long process", (Throwable)e);
            }
        }
    }
}

